/* The Knight's Tour                                                         */
/* Copyright C. Broadribb 1995                                               */
/* 28/4/95                                                                   */

#include <stdio.h>
#include <string.h>
#include <conio.h>
#include <stdlib.h>
#include <time.h>
#include <dos.h>

/* The following macro returns a random number between min and max.  If max is < 0 then it returns 0. */
#define rnd( min, max ) (max > 0)?((rand() % (int)(((max)+1) - (min))) + (min)):0

/* These define the desired colours. */
#define light 7          /* yellow */
#define dark 6            /* brown  */
#define knightcolour 0    /* black  */
#define markercolour 0    /* black  */
#define normalb 0         /* black  */
#define normalf 7         /* white  */

/* This macro returns true if position y, x is a light coloured square.  */
#define lightsquare (y % 2 == 0 && x % 2 == 0) || (y % 2 == 1 && x % 2 == 1)

enum boolean { false, true };

void initialiseboard (char [9][9], int [9][9]);
void drawboard (enum boolean);
void showmoves (int [9][9]);
void drawknight (char [9][9], int, int, int);
void marksquare (char [9][9], int, int);
void inputstart (int *, int *);
void choosestart (int *, int *);
void choosemove (int [9][9], int *, int *);
enum boolean keypress ();
enum boolean askuser();
void beepandpause ();

void main (int argc, char *argv[])
{
	char board [9][9];
	int history [9][9];
	int xpos, ypos, oldx, oldy, moveno;
	enum boolean autoplay, playgame, runprog;

	/* This initialises path with the direction indicators to generate a path around the chess board. */
	int path [9][9] =
	{
		0,0,0,0,0,0,0,0,0,
		0,1,2,8,2,1,8,1,2,
		0,7,2,4,4,7,7,4,2,
		0,7,2,8,1,5,4,5,2,
		0,1,5,5,3,6,2,1,5,
		0,1,8,6,6,4,4,2,4,
		0,6,2,3,1,4,8,6,2,
		0,8,8,6,3,6,8,6,3,
		0,7,5,7,5,7,4,5,5
	};

	/* Seed the random number generator. */
	srand( (unsigned)time( NULL ) );

	runprog = true;
	autoplay = false;
	if (argc > 1 && stricmp (argv[1], "/a") == 0)
		autoplay = true;
	do
	{
		moveno = 0;
		playgame = true;
		initialiseboard (board, history);

		if (autoplay == true)
			choosestart (&xpos, &ypos);
		else
			inputstart (&xpos, &ypos);
		drawboard (autoplay);
		do
		{
			moveno ++;
			drawknight (board, xpos, ypos, moveno);
			history [ypos][xpos] = moveno;
			if (moveno == 64)
			{
				playgame = false;
				if (autoplay == false)
				{
					showmoves (history);
					runprog = askuser ();
				}
			}
			else
			{
				oldx = xpos;
				oldy = ypos;
				choosemove (path, &xpos, &ypos);
				marksquare (board, oldx, oldy);
			}

			if (autoplay == true)
				runprog = (enum boolean) !keypress();

		} while (runprog && playgame);

	} while (runprog);
	clrscr();
}


void initialiseboard (char board [9][9], int history [9][9])
{
	/* Post: board is initialised to blank spaces.  History is           */
	/* initialised to zeros.                                             */

	int y, x;

	for (y = 1; y <= 8; y ++)
		for (x = 1; x <= 8; x ++)
		{
			board [y][x] = ' ';
			history [y][x] = 0;
		}

}


void drawboard (enum boolean autoplay)
{
	/* Post:  the chess board is drawn on the screen.                    */

	int x, y;

	clrscr();
	textbackground (normalb);
	textcolor (normalf);
	for (y = 1; y <= 8; y ++)
	{
		for (x = 1; x <= 8; x ++)
		{
			if (lightsquare)
				textcolor (light);
			else
				textcolor (dark);
			cputs ("");
		}
		cputs ("\r\n");

		for (x = 1; x <= 8; x ++)
		{
			if (lightsquare)
				textcolor (light);
			else
				textcolor (dark);
			cputs ("");
		}
		cputs ("\r\n");

		for (x = 1; x <= 8; x ++)
		{
			if (lightsquare)
				textcolor (light);
			else
				textcolor (dark);
			cputs ("");
		}
		cputs ("\r\n");
	}
	textbackground (normalb);
	textcolor (normalf);
	gotoxy (65,1);
	cputs ("Knight's Tour");
	gotoxy (65,2);
	cputs ("Copyright");
	gotoxy (65,3);
	cputs ("C. Broadribb");
	gotoxy (65,4);
	cputs ("1995");
	if (autoplay == true)
	{
		gotoxy (65,10);
		cputs ("Press any key");
		gotoxy (65,11);
		cputs ("to stop.");
	}
}


void showmoves (int history [9][9])
{
	/* Pre:  history has the move number for each chess square.          */
	/* Post: The move number for each square is displayed on the screen. */

	int y, x;
	char move [3];

	for (y = 1; y <= 8; y ++)
		for (x = 1; x <= 8; x ++)
		{
			if (lightsquare)
			{
				textcolor (light);
				textbackground (light);
			}
			else
			{
				textcolor (dark);
				textbackground (dark);
			}
			/* Wipe out cross */
			gotoxy ((x-1)*7+3, (y-1)*3+1);
			cputs ("");
			gotoxy ((x-1)*7+3, (y-1)*3+2);
			cputs ("");
			gotoxy ((x-1)*7+3, (y-1)*3+3);
			cputs ("");

			/* Display move number */
			gotoxy ((x-1)*7+3, (y-1)*3+2);
			textcolor (knightcolour);
			sprintf (move, "%0.2i",history [y][x]);
			cputs (move);
	       }

	textbackground (normalb);
	textcolor (normalf);
}


void drawknight (char board [9][9], int x, int y, int moveno)
{
	/* Pre:  x and y are valid coordinates on the chess board.           */
	/* Post: The knight is drawn on the screen at position y,x and the   */
	/* row, column and move number are displayed at the bottom of the    */
	/* chess board.                                                      */
	
	char info [80];
	textbackground (normalb);
	textcolor (normalf);
	gotoxy (1,25);
	sprintf (info,"Row:  %i  Column:  %i  Move number:  %i",y,x,moveno);
	cputs (info);

	board [y] [x] = 'K';
	textcolor (knightcolour);
	if (lightsquare)
		textbackground (light);
	else
		textbackground (dark);

	gotoxy ((x-1)*7+4, (y-1)*3+1);
	cputs ("");
	gotoxy ((x-1)*7+3, (y-1)*3+2);
	cputs ("");
	gotoxy ((x-1)*7+3, (y-1)*3+3);
	cputs ("/");
	gotoxy ((x-1)*7+5, (y-1)*3+3);
	cputs ("\\");

	beepandpause();
	textbackground (normalb);
	textcolor (normalf);

}


void marksquare (char board [9][9], int x, int y)
{
	/* Pre:  x and y are valid coordinates on the chess board.           */
	/* Post: A marker is displayed at position y,x.                      */
	
	board [y] [x] = '*';

	/* Wipe out figure */
	if (lightsquare)
	{
		textcolor (light);
		textbackground (light);
	}
	else
	{
		textcolor (dark);
		textbackground (dark);
	}
	gotoxy ( (x-1)*7+4, (y-1)*3+1);
	cputs ("");
	gotoxy ( (x-1)*7+3, (y-1)*3+2);
	cputs ("");

	/* Draw cross */
	textcolor (markercolour);
	gotoxy ((x-1)*7+3, (y-1)*3+1);
	cputs ("*");
	gotoxy ((x-1)*7+5, (y-1)*3+1);
	cputs ("*");
	gotoxy ((x-1)*7+4, (y-1)*3+2);
	cputs ("*");
	gotoxy ((x-1)*7+3, (y-1)*3+3);
	cputs ("*");
	gotoxy ((x-1)*7+5, (y-1)*3+3);
	cputs ("*");

	textbackground (normalb);
	textcolor (normalf);
}


void inputstart (int *xpos, int *ypos)
{
	/* Post: xpos and ypos contain x and y coordinates input by the      */
	/* user.                                                             */
	/* If the user enters invalid input for the x or y coordinates then  */
	/* an error message is displayed and the user must enter them again. */
	
	int x, y;
	char input [10];

	clrscr();
	puts ("*****   *   *   *****");
	puts ("  *     *   *   *    ");
	puts ("  *     *****   *****");
	puts ("  *     *   *   *    ");
	puts ("  *     *   *   *****");
	puts ("\r\n");

	puts ("*   *   *   *   *****    ****   *   *   *****    *    ****");
	puts ("*  *    **  *     *     *       *   *     *     **   *    ");
	puts ("* *     * * *     *     *  **   *****     *           *** ");
	puts ("*  *    *  **     *     *   *   *   *     *              *");
	puts ("*   *   *   *   *****    ***    *   *     *          **** ");
	puts ("\r\n");

	puts ("*****    ***    *   *   **** ");
	puts ("  *     *   *   *   *   *   *");
	puts ("  *     *   *   *   *   **** ");
	puts ("  *     *   *   *   *   *   *");
	puts ("  *      ***     ***    *   *         Copyright C.A. Broadribb 1995");


	puts ("\r\nYou must enter the starting position for the knight on the chessboard.");
	puts ("This is a row number from 1 to 8 and a column number from 1 to 8, eg, 4,5");

	do
	{
		printf ("Enter starting position (row, column):  ");
		
		/* Empty the keyboard buffer. */
		fflush (stdin);
		
		gets (input);
		puts ("");
		sscanf (input, "%i,%i", &y, &x);
		if (y < 1 || y > 8)
			puts ("Row must be from 1 to 8.");
		if (x < 1 || x > 8)
			puts ("Column must be from 1 to 8.");
	} while (x < 1 || x > 8 || y < 1 || y > 8);
	*xpos = x;
	*ypos = y;
}


void choosestart (int *xpos, int *ypos)
{
	/* Post: xpos and ypos have randomly generated values between 1 and  */
	/* 8.                                                                */

	*xpos = (rnd (1,8));
	*ypos = (rnd (1,8));
}


void choosemove (int path [9][9], int *xpos, int *ypos)
{
	/* Pre:  Path contains valid direction indicators (numbers between   */
	/* 1 and 8).  Xpos and ypos are valid coordinates on the chess board.*/
	/* Post:  xpos and ypos hold the new coordinates.                    */

	int direction;
	direction = path [*ypos][*xpos];
	switch (direction)
	{
		case 1: *xpos += 1; *ypos += 2; break;
		case 2: *xpos -= 1; *ypos += 2; break;
		case 3: *xpos -= 2; *ypos += 1; break;
		case 4: *xpos -= 2; *ypos -= 1; break;
		case 5: *xpos -= 1; *ypos -= 2; break;
		case 6: *xpos += 1; *ypos -= 2; break;
		case 7: *xpos += 2; *ypos -= 1; break;
		case 8: *xpos += 2; *ypos += 1; break;
	};
}


enum boolean keypress ()
{
	/* Post: if a key has been pressed then keypress returns true.       */
	/* Otherwise, keypress returns false.                                */

	int key;

	key = kbhit();
	if (key)                /* kbhit returns 0 if no key is pressed or a number if a key is pressed. */
	{
		getch();        /* Read the key press and discard it */
		return (true);
	}
	else
		return (false);

}


enum boolean askuser()
{
	/* Post: if the user wishes to run another tour, askuser returns     */
	/* true.  Otherwise, askuser returns false.                          */

	char answer;

	do
	{
		gotoxy (65,10);

		/* Empty the keyboard buffer */
		fflush (stdin);
		cputs ("Another tour?");
		gotoxy (65,11);
		cputs ("Y/N?  ");
		scanf ("%c", &answer);

	} while (answer != 'n' && answer != 'N' && answer != 'y' && answer != 'Y');

	if (answer == 'y' || answer == 'Y')
	       return (true);
	else
	       return (false);
}


void beepandpause (void)
{
	sound (60);
	delay (500);
	nosound();
	delay (500);
}
